//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

import wesual.Ui 1.0

Item {
    id : colorEditor

    property color color : "white"

    property bool indeterminate : false

    property alias additionalControls : controls.sourceComponent

    signal updateRequested(color newColor)

    function setLightness(lightness) {
        colorEditor.updateRequested(
                    Qt.hsla(color.hslHue, color.hslSaturation,
                            lightness, 1.0));
    }

    function setHueSaturation(hue, saturation) {
        colorEditor.updateRequested(
                    Qt.hsla(hue, saturation, clamp(color.hslLightness), 1.0));
    }

    function clamp(value) {
        return Math.max(.002, Math.min(.998, value));
    }

    implicitWidth : 312
    implicitHeight : 302

    ShaderEffect {
        id : hueSaturationPicker

        readonly property real lightness : clamp(color.hslLightness)

        width : 256
        height : 256
        x : 3
        y : 3

        mesh : GridMesh {
            resolution : Qt.size(18, 12)
        }

        UiBorder {
            anchors.fill : parent
            style : UiBorder.Solid
            borderWidth : 1
            color : UiColors.getColor(UiColors.Black, 0.1)
        }

        MouseArea {
            function updateValue(x, y) {
                var mapped = mapToItem(parent, x, y);
                setHueSaturation(clamp(mapped.x / parent.width),
                                 clamp(1.0 - mapped.y / parent.height));
            }

            anchors.fill : parent
            anchors.margins : -4

            acceptedButtons : Qt.LeftButton
            preventStealing : true

            onPressed : updateValue(mouse.x, mouse.y)
            onPositionChanged : updateValue(mouse.x, mouse.y)
        }

        Item {
            visible : !colorEditor.indeterminate
            x : Math.round(
                    1 + clamp(color.hslHue) * (hueSaturationPicker.width - 1))
            y : Math.round(
                    1 + (1.0 - color.hslSaturation) *
                    (hueSaturationPicker.height - 1))

            Image {
                source : "qrc:/ui/images/color-hue-saturation-handle.png"
                x : -7
                y : -7
            }
        }

        vertexShader : "
            attribute highp vec4 qt_Vertex;
            attribute highp vec2 qt_MultiTexCoord0;

            uniform highp mat4 qt_Matrix;

            uniform lowp float lightness;

            varying lowp vec4 v_Color;

            void main() {
                lowp float hue = qt_MultiTexCoord0.x * 6.0;
                lowp float saturation = 1.0 - qt_MultiTexCoord0.y;

                lowp float chroma =
                    (1.0 - abs(2.0 * lightness - 1.0)) * saturation;
                lowp float x = chroma * (1.0 - abs(mod(hue, 2.0) - 1.0));
                lowp vec3 comp = vec3(chroma, x, 0.0);

                lowp vec3 branches[6];
                branches[0] = comp.xyz;
                branches[1] = comp.yxz;
                branches[2] = comp.zxy;
                branches[3] = comp.zyx;
                branches[4] = comp.yzx;
                branches[5] = comp.xzy;

                int idx = int(floor(min(5.9, max(0.1, hue))));
                lowp vec3 rgb = branches[idx] + (lightness - chroma / 2.0);
                v_Color = vec4(rgb, 1.0);

                gl_Position = qt_Matrix * qt_Vertex;
            }"

        fragmentShader : "
            varying lowp vec4 v_Color;

            uniform lowp float qt_Opacity;

            void main() {
                gl_FragColor = v_Color * qt_Opacity;
            }"
    }

    ShaderEffect {
        id : lightnessPicker

        readonly property real hue : clamp(color.hslHue) * 6
        readonly property real saturation : clamp(color.hslSaturation)

        width : 22
        height : 256
        x : 271
        y : 3
        mesh : GridMesh {
            resolution : Qt.size(1, 5)
        }

        UiBorder {
            anchors.fill : parent
            style : UiBorder.Solid
            borderWidth : 1
            color : UiColors.getColor(UiColors.Black, 0.1)
        }

        MouseArea {
            function updateValue(y) {
                var mapped = mapToItem(parent, 0, y);
                setLightness(clamp(1.0 - mapped.y / parent.height));
            }

            anchors {
                fill : parent
                leftMargin : -5
                rightMargin : -20
                topMargin : -5
                bottomMargin : -10
            }

            acceptedButtons : Qt.LeftButton
            preventStealing : true

            onPressed : updateValue(mouse.y)
            onPositionChanged : updateValue(mouse.y)
        }

        Item {
            anchors {
                left : lightnessPicker.right
                leftMargin : -10
            }
            visible : !colorEditor.indeterminate
            y : Math.round(
                    (1.0 - color.hslLightness) * (lightnessPicker.height - 1))

            Image {
                source : "qrc:/ui/images/color-lightness-handle.png"
                y : -8

                layer.enabled : true
                layer.smooth : true

                layer.effect : ShaderEffect {
                    property color color : colorEditor.color

                    fragmentShader : "
                        uniform lowp sampler2D source;
                        uniform lowp vec4 color;

                        uniform lowp float qt_Opacity;
                        varying highp vec2 qt_TexCoord0;
                        void main() {
                            lowp vec4 sample = texture2D(source, qt_TexCoord0);
                            lowp vec3 ctrl = vec3(sample.r);
                            gl_FragColor = vec4(
                                mix(ctrl, color.rgb, sample.g), sample.a)
                                * qt_Opacity;
                        }"
                }
            }
        }

        vertexShader : "
            attribute highp vec4 qt_Vertex;
            attribute mediump vec2 qt_MultiTexCoord0;

            uniform highp mat4 qt_Matrix;

            uniform lowp float hue;
            uniform lowp float saturation;

            varying mediump vec4 v_Color;

            void main() {
                gl_Position = qt_Matrix * qt_Vertex;
                lowp float lightness = 1.0 - qt_MultiTexCoord0.y;

                lowp float chroma =
                    (1.0 - abs(2.0 * lightness - 1.0)) * saturation;
                lowp float x = chroma * (1.0 - abs(mod(hue, 2.0) - 1.0));
                lowp vec3 comp = vec3(chroma, x, 0.0);

                lowp vec3 branches[6];
                branches[0] = comp.xyz;
                branches[1] = comp.yxz;
                branches[2] = comp.zxy;
                branches[3] = comp.zyx;
                branches[4] = comp.yzx;
                branches[5] = comp.xzy;

                lowp vec3 rgb = branches[int(floor(min(5.9, hue)))] +
                    (lightness - chroma / 2.0);

                v_Color = vec4(rgb, 1.0);
            }"

        fragmentShader : "
            varying mediump vec4 v_Color;

            uniform lowp float qt_Opacity;

            void main() {
                gl_FragColor = v_Color * qt_Opacity;
            }"
    }

    Row {
        y : 268
        x : 3

        spacing : 6

        Rectangle {
            id : colorPatch

            height : 30
            width : 84
            color : colorEditor.color

            UiBorder {
                anchors.fill : parent
                color : UiColors.getColor(UiColors.Black, 0.1)
                borderWidth : 1
                style : UiBorder.Solid
            }
        }

        UiLineEdit {
            anchors {
                verticalCenter : colorPatch.verticalCenter
            }
            text : colorEditor.indeterminate ? "" : colorEditor.color.toString()
            bindable : true
            validator : RegExpValidator {
                regExp : /^#?(?:[a-f0-9]{3}|[a-f0-9]{6})$/i
            }
            onUpdateRequested : {
                if (newValue[0] !== "#") {
                    newValue = "#" + newValue;
                }

                colorEditor.updateRequested(newValue);
            }
        }

        Loader {
            id : controls
            visible : !colorEditor.indeterminate
        }
    }
}
